<?php
session_start(); // Start session at the very beginning

// Use fixed seed for random number generator to ensure consistent results
mt_srand(42);

// Check if we have cached results for the same input
if (isset($_SESSION['rekomendasi']) && 
    isset($_SESSION['input_data']) && 
    $_SESSION['input_data'] == $_POST) {
    
    // Use cached results
    $ranking = $_SESSION['rekomendasi'];
    extract($_SESSION['input_data']); // Extract POST data as variables
} else {
    // Process as normal since we don't have cached results
    include 'koneksi.php';

    // =============================================
    // FUNGSI UTAMA (DIPERBAIKI)
    // =============================================

    function hitungJarak($point1, $point2) {
        return sqrt(
            pow($point1['harga'] - $point2['harga'], 2) +
            pow($point1['trend'] - $point2['trend'], 2) +
            pow($point1['aktivitas'] - $point2['aktivitas'], 2) +
            pow($point1['kecocokan'] - $point2['kecocokan'], 2)
        );
    }

    // Inisialisasi centroid berdasarkan gaya fashion yang ada di database
    function initCentroids($produk) {
        $centroids = [];
        $gaya_uniq = [];
        
        // Identifikasi semua gaya fashion unik dari produk
        foreach ($produk as $item) {
            if (!in_array($item['style'], $gaya_uniq)) {
                $gaya_uniq[] = $item['style'];
            }
        }
        
        // Untuk setiap gaya fashion, hitung centroidnya
        $i = 1;
        foreach ($gaya_uniq as $gaya) {
            $items_gaya = array_filter($produk, function($item) use ($gaya) {
                return $item['style'] === $gaya;
            });
            
            $count = count($items_gaya);
            if ($count > 0) {
                $sum_harga = array_sum(array_column($items_gaya, 'harga'));
                $sum_trend = array_sum(array_column($items_gaya, 'trend'));
                $sum_aktivitas = array_sum(array_column($items_gaya, 'aktivitas'));
                $sum_kecocokan = array_sum(array_column($items_gaya, 'kecocokan'));
                
                $centroids[$i] = [
                    'harga' => $sum_harga / $count,
                    'trend' => $sum_trend / $count,
                    'aktivitas' => $sum_aktivitas / $count,
                    'kecocokan' => $sum_kecocokan / $count,
                    'style' => $gaya
                ];
                $i++;
            }
        }
        
        return $centroids;
    }

    // =============================================
    // PROSES UTAMA (DIPERBAIKI)
    // =============================================

    // 1. Ambil data dari form dengan validasi
    $gender = $_POST['gender'] ?? '';
    $warna = $_POST['warna'] ?? '';
    $harga_input = (float)($_POST['harga'] ?? 0);
    $trend_input = (float)($_POST['trend'] ?? 0);
    $aktivitas_input = (float)($_POST['aktivitas_penting'] ?? 0);
    $kecocokan_input = (float)($_POST['gaya'] ?? 0);
    $aktivitas_harian = $_POST['aktivitas'] ?? '';

    // ===== [TAMBAHKAN KODE INI UNTUK SIMPAN PREFERENSI USER] =====
    // Generate a unique session ID for this user if it doesn't exist
    if (!isset($_SESSION['user_session_id'])) {
        $_SESSION['user_session_id'] = uniqid('user_', true);
    }

    // Simpan data preferensi user ke tabel 'user_preferences'
    // SESUAIKAN DENGAN STRUKTUR TABEL YANG ADA
    $sql_save_pref = "INSERT INTO user_preferences 
                    (user_session, gender, aktivitas_harian, harga, trend, aktivitas_skor, kecocokan) 
                    VALUES (?, ?, ?, ?, ?, ?, ?)";

    $stmt_save_pref = $db->prepare($sql_save_pref);
    $stmt_save_pref->bind_param("sssiiii", 
        $_SESSION['user_session_id'], 
        $gender, 
        $aktivitas_harian,
        $harga_input,
        $trend_input,
        $aktivitas_input,
        $kecocokan_input
    );

    $stmt_save_pref->execute();
    $stmt_save_pref->close();
    // ===== [AKHIR DARI KODE YANG DITAMBAHKAN] =====

    // 2. Ambil data produk dengan prepared statement
    $stmt = $db->prepare("SELECT * FROM produk WHERE gender = ? AND warna = ? AND aktivitas LIKE ?");
    $like_aktivitas = "%$aktivitas_harian%";
    $stmt->bind_param("sss", $gender, $warna, $like_aktivitas);
    $stmt->execute();
    $result = $stmt->get_result();

    $produk = [];
    while ($row = $result->fetch_assoc()) {
        $produk[] = [
            'id' => $row['id_produk'],
            'nama' => $row['nama_produk'],
            'style' => $row['style'],
            'harga' => (float)$row['skor_harga'],
            'trend' => (float)$row['skor_trend'],
            'aktivitas' => (float)$row['skor_aktivitas'],
            'kecocokan' => (float)$row['skor_kecocokangaya'],
            'cluster' => 0
        ];
    }

    // =============================================
    // PROSES K-MEANS CLUSTERING
    // =============================================

    // 3. Inisialisasi centroid berdasarkan gaya fashion yang ada
    $centroids = initCentroids($produk);
    $k = count($centroids);
    
    // 4. Proses clustering - Tetapkan setiap produk ke cluster berdasarkan gaya fashion
    foreach ($produk as $idx => &$item) {
        $jarak_terdekat = PHP_FLOAT_MAX;
        $cluster_terdekat = 0;
        
        foreach ($centroids as $c_idx => $centroid) {
            // Pastikan produk hanya masuk ke cluster dengan gaya yang sama
            if ($item['style'] === $centroid['style']) {
                $jarak = hitungJarak($item, $centroid);
                if ($jarak < $jarak_terdekat) {
                    $jarak_terdekat = $jarak;
                    $cluster_terdekat = $c_idx;
                }
            }
        }
        
        if ($cluster_terdekat > 0) {
            $item['cluster'] = $cluster_terdekat;
        }
    }
    unset($item);

    // 5. Tentukan cluster terdekat dengan user
    $user_point = [
        'harga' => $harga_input,
        'trend' => $trend_input,
        'aktivitas' => $aktivitas_input,
        'kecocokan' => $kecocokan_input
    ];

    $jarak_terdekat = PHP_FLOAT_MAX;
    $user_cluster = 0;
    $user_cluster_gaya = '';
    $jarak_cluster = [];

    foreach ($centroids as $idx => $centroid) {
        $jarak = hitungJarak($user_point, $centroid);
        $jarak_cluster[$idx] = $jarak;
        
        if ($jarak < $jarak_terdekat) {
            $jarak_terdekat = $jarak;
            $user_cluster = $idx;
            $user_cluster_gaya = $centroid['style'];
        }
    }

    // 6. Filter produk dari cluster terdekat
    $produk_cluster = array_filter($produk, function($item) use ($user_cluster_gaya) {
        return $item['style'] === $user_cluster_gaya;
    });

    // =============================================
    // PROSES MOORA
    // =============================================

    // 7.1. Normalisasi matriks keputusan
    $sum_sq = ['harga' => 0.0001, 'trend' => 0.0001, 'aktivitas' => 0.0001, 'kecocokan' => 0.0001];

    foreach ($produk_cluster as $item) {
        $sum_sq['harga'] += pow($item['harga'], 2);
        $sum_sq['trend'] += pow($item['trend'], 2);
        $sum_sq['aktivitas'] += pow($item['aktivitas'], 2);
        $sum_sq['kecocokan'] += pow($item['kecocokan'], 2);
    }

    // 7.2. Hitung bobot kriteria
    $total_bobot = $harga_input + $trend_input + $aktivitas_input + $kecocokan_input;
    if ($total_bobot == 0) $total_bobot = 1;

    $bobot = [
        'harga' => $harga_input / $total_bobot,
        'trend' => $trend_input / $total_bobot,
        'aktivitas' => $aktivitas_input / $total_bobot,
        'kecocokan' => $kecocokan_input / $total_bobot
    ];

    // 7.3. Hitung skor optimasi (Yi)
    $ranking = [];
    foreach ($produk_cluster as $item) {
        $harga_norm = $item['harga'] / sqrt($sum_sq['harga']);
        $trend_norm = $item['trend'] / sqrt($sum_sq['trend']);
        $aktivitas_norm = $item['aktivitas'] / sqrt($sum_sq['aktivitas']);
        $kecocokan_norm = $item['kecocokan'] / sqrt($sum_sq['kecocokan']);
        
        $benefit = ($trend_norm * $bobot['trend']) +
                   ($aktivitas_norm * $bobot['aktivitas']) +
                   ($kecocokan_norm * $bobot['kecocokan']);
        
        $cost = $harga_norm * $bobot['harga'];
        $yi = $benefit - $cost;
        
        $ranking[] = [
            'id' => $item['id'],
            'nama' => $item['nama'],
            'yi' => $yi,
            'harga' => $item['harga'],
            'trend' => $item['trend'],
            'aktivitas' => $item['aktivitas'],
            'kecocokan' => $item['kecocokan'],
            'gender' => $gender,
            'warna' => $warna,
            'style' => $item['style'],
            'cluster' => $item['cluster']
        ];
    }

    // ===== [TAMBAHKAN KODE INI UNTUK SIMPAN REKOMENDASI SISTEM] =====
    // Simpan hasil rekomendasi (ranking) ke database
    $rank_position = 1;
    foreach ($ranking as $item) {
        // Hanya simpan 10 rekomendasi teratas untuk efisiensi
        if ($rank_position > 10) {
            break;
        }

        $sql_save_rec = "INSERT INTO system_recommendations 
                        (user_session, product_id, rank_position, moora_score) 
                        VALUES (?, ?, ?, ?)";
        
        $stmt_save_rec = $db->prepare($sql_save_rec);
        $stmt_save_rec->bind_param("siid", 
            $_SESSION['user_session_id'], 
            $item['id'], 
            $rank_position, 
            $item['yi']
        );
        
        $stmt_save_rec->execute();
        $stmt_save_rec->close();
        
        $rank_position++;
    }
    // ===== [AKHIR DARI KODE YANG DITAMBAHKAN] =====

    // Urutkan berdasarkan Yi tertinggi
    usort($ranking, function($a, $b) {
        return $b['yi'] <=> $a['yi'];
    });

    // Store results in session for future use
    $_SESSION['rekomendasi'] = $ranking;
    $_SESSION['input_data'] = $_POST;
}

// =============================================
// AMBIL DATA RATING USER
// =============================================

// AMBIL SEMUA RATING USER UNTUK PRODUK YANG DIREKOMENDASIKAN SEKALIGUS - SEBELUM LOOP
$user_ratings = []; // Buat array kosong untuk menyimpan hasil
if (isset($_SESSION['user_session_id']) && !empty($ranking)) {
    include 'koneksi.php';
    
    // Kumpulkan semua ID produk dari hasil rekomendasi
    $product_ids = array_column($ranking, 'id');
    // Buat placeholder untuk prepared statement (?, ?, ?, ...)
    $placeholders = implode(',', array_fill(0, count($product_ids), '?'));
    
    // Query untuk mengambil semua rating user untuk produk-produk ini
    $ratings_sql = "SELECT product_id, user_rating FROM ground_truth WHERE user_session = ? AND product_id IN ($placeholders)";
    $ratings_stmt = $db->prepare($ratings_sql);
    
    // Bangun parameter untuk bind_param: string 's' untuk session, diikuti banyak 'i' untuk product_ids
    $param_types = 's' . str_repeat('i', count($product_ids));
    $params = array_merge([$_SESSION['user_session_id']], $product_ids);
    
    // Bind parameter secara dinamis menggunakan referensi
    $ref_params = [];
    foreach ($params as $key => $value) {
        $ref_params[$key] = &$params[$key];
    }
    call_user_func_array([$ratings_stmt, 'bind_param'], array_merge([$param_types], $ref_params));
    
    $ratings_stmt->execute();
    
    // Ganti get_result() dengan bind_result()
    $ratings_stmt->bind_result($product_id, $user_rating_val);
    
    while ($ratings_stmt->fetch()) {
        $user_ratings[$product_id] = $user_rating_val;
    }
    $ratings_stmt->close();
    $db->close();
}

// =============================================
// TAMPILKAN HASIL
// =============================================
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <title>Hasil Rekomendasi Fashion</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- SweetAlert2 -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        :root {
            --pink-primary: #ff66b2;
            --pink-light: #ff99cc;
            --pink-lighter: #ffcce6;
            --pink-dark: #e60073;
            --pink-darker: #b30059;
        }
        
        body {
            background-color: #fff0f5;
            padding-bottom: 60px;
        }
        
        .navbar, .card-header, .btn-primary {
            background-color: var(--pink-primary) !important;
            border-color: var(--pink-dark) !important;
        }
        
        .btn-primary:hover {
            background-color: var(--pink-dark) !important;
            border-color: var(--pink-darker) !important;
        }
        
        .alert-info {
            background-color: var(--pink-lighter);
            border-color: var(--pink-light);
            color: var(--pink-darker);
        }
        
        .produk-card {
            transition: transform 0.3s;
            margin-bottom: 20px;
            border: 1px solid var(--pink-light);
            background-color: white;
            height: 100%;
        }
        
        .produk-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(255, 102, 178, 0.2);
        }
        
        .badge-gender {
            background-color: var(--pink-dark);
            color: white;
            display: block;
            margin-bottom: 5px;
            width: fit-content;
        }
        
        .badge-style {
            background-color: var(--pink-light);
            color: var(--pink-darker);
            display: block;
            width: fit-content;
        }
        
        .badge-warna {
            background-color: var(--pink-primary);
            color: white;
            display: block;
            margin-bottom: 5px;
            width: fit-content;
        }
        
        h1, h2, h3, h4, h5, h6 {
            color: var(--pink-darker);
        }
        
        .btn-back {
            background-color: var(--pink-primary);
            border-color: var(--pink-dark);
            color: white;
            margin-top: 30px;
            padding: 10px 20px;
        }
        
        .btn-back:hover {
            background-color: var(--pink-darker);
            color: white;
        }
        
        .card-body {
            display: flex;
            flex-direction: column;
        }
        
        .card-title {
            font-size: 1.1rem;
        }
        
        .badge-container {
            margin-bottom: 15px;
        }
        
        .footer-button {
            text-align: center;
            margin-top: 40px;
        }
        
        .rating-buttons {
            margin-top: 10px;
            padding-top: 10px;
            border-top: 1px solid #eee;
            text-align: center;
        }
        
        .btn-rating {
            font-size: 0.8rem;
            padding: 3px 8px;
            margin: 0 2px;
        }
        
        .rated-badge {
            background-color: var(--pink-primary);
            color: white;
        }
        
        .rating-container {
            display: flex;
            flex-direction: column;
            align-items: center;
        }
        
        .rating-text {
            margin-bottom: 5px;
            font-weight: 500;
        }
        
        .rating-actions {
            display: flex;
            justify-content: center;
            gap: 5px;
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark mb-4">
        <div class="container">
            <a class="navbar-brand" href="#">STYLO REKOMENDASI</a>
        </div>
    </nav>

    <div class="container py-4">
        <h1 class="text-center mb-4">Hasil Rekomendasi Fashion</h1>
        
        <div class="alert alert-info text-center">
            <strong>Total Produk Direkomendasikan:</strong> <?= count($ranking) ?>
            <br>
            <small>Bantu kami menjadi lebih baik! Beri rating pada produk yang direkomendasikan.</small>
        </div>

        <div class="row">
            <?php foreach ($ranking as $i => $produk): 
                // Sekarang cek rating dari array yang sudah kita ambil, bukan dari database
                $already_rated = array_key_exists($produk['id'], $user_ratings);
                $user_rating = $already_rated ? $user_ratings[$produk['id']] : null;
            ?>
                <div class="col-md-4 col-lg-3 mb-4">
                    <div class="card produk-card h-100">
                        <div class="card-header text-white">
                            <h5 class="card-title mb-0">
                                <?= htmlspecialchars($produk['nama']) ?>
                            </h5>
                        </div>
                        <div class="card-body">
                            <div class="badge-container">
                                <span class="badge badge-gender">
                                    Gender: <?= htmlspecialchars($produk['gender'] ?? $gender) ?>
                                </span>
                                <span class="badge badge-warna">
                                    Warna: <?= htmlspecialchars($produk['warna'] ?? $warna) ?>
                                </span>
                                <span class="badge badge-style">
                                    Gaya: <?= htmlspecialchars($produk['style'] ?? '') ?>
                                </span>
                                <?php if ($already_rated): ?>
                                <span class="badge rated-badge">
                                    <?= $user_rating ? '👍 Disukai' : '👎 Tidak Disukai' ?>
                                </span>
                                <?php endif; ?>
                            </div>
                            <a href="https://shopee.co.id/search?keyword=<?= urlencode($produk['nama']) ?>" 
                               target="_blank" class="btn btn-primary mt-3">
                                Lihat Produk
                            </a>
                            <!-- TOMBOL RATING DENGAN AJAX -->
                            <div class="rating-buttons mt-2">
                                <small class="rating-text">Apakah produk ini relevan?</small>
                                
                                <div class="rating-actions">
                                    <button onclick="submitRating(<?= $produk['id'] ?>, 1, '<?= $_SESSION['user_session_id'] ?>')" 
                                            class="btn btn-sm btn-success btn-rating <?= $already_rated && $user_rating == 1 ? 'disabled' : '' ?>"
                                            <?= $already_rated && $user_rating == 1 ? 'disabled' : '' ?>>
                                        👍 Ya
                                    </button>
                                    <button onclick="submitRating(<?= $produk['id'] ?>, 0, '<?= $_SESSION['user_session_id'] ?>')" 
                                            class="btn btn-sm btn-danger btn-rating <?= $already_rated && $user_rating == 0 ? 'disabled' : '' ?>"
                                            <?= $already_rated && $user_rating == 0 ? 'disabled' : '' ?>>
                                        👎 Tidak
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- Tombol Kembali -->
        <div class="text-center mt-5">
            <a href="form.php" class="btn btn-lg btn-back">
                ← Kembali ke Form Rekomendasi
            </a>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    function submitRating(productId, rating, userSession) {
        // Tampilkan loading popup
        Swal.fire({
            title: 'Menyimpan rating...',
            text: 'Harap tunggu sebentar',
            allowOutsideClick: false,
            didOpen: () => {
                Swal.showLoading()
            }
        });

        // Kirim data via AJAX
        fetch('save_rating.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `product_id=${productId}&rating=${rating}&user_session=${userSession}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Tutup loading popup
                Swal.close();
                
                // Tampilkan success popup
                Swal.fire({
                    icon: 'success',
                    title: 'Terima kasih!',
                    text: data.message,
                    timer: 2000,
                    showConfirmButton: false
                }).then(() => {
                    // Refresh halaman untuk update tampilan
                    window.location.reload();
                });
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Oops...',
                    text: data.message
                });
            }
        })
        .catch(error => {
            Swal.close();
            Swal.fire({
                icon: 'error',
                title: 'Oops...',
                text: 'Terjadi kesalahan: ' + error
            });
        });
    }
    </script>
</body>
</html>